<?php
/*
Plugin Name: AI Bulk ChatGPT Silo
Description: Tạo bài viết hàng loạt từ danh sách từ khóa bằng API ChatGPT (Queue nền, đa ngôn ngữ, license serial + email + domain, quản lý số bài viết, tạo link nội bộ silo, tone & kiểu bài).
Version: 4.2
Author: Fame Media
Text Domain: ai-bulk-chatgpt-silo
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AI_Bulk_ChatGPT_Silo {

    const OPTION_SETTINGS = 'abcgs_settings';
    const OPTION_QUEUE    = 'abcgs_queue';
    const OPTION_SILO_LOG = 'abcgs_silo_log';
    const OPTION_LICENSE  = 'abcgs_license';

    const CRON_HOOK       = 'abcgs_process_queue_event';

    // ✅ License API URL cố định theo yêu cầu
    const LICENSE_API_URL = 'https://famemedia.vn/licenses/license-api.php';

    public function __construct() {
        @set_time_limit( 300 );
        @ini_set( 'max_execution_time', 300 );

        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( self::CRON_HOOK, array( $this, 'process_queue' ) );
    }

    public function add_admin_menu() {
        add_menu_page(
            'AI Bulk ChatGPT Silo',
            'AI Bulk Silo',
            'manage_options',
            'abcgs-ai-bulk-silo',
            array( $this, 'render_admin_page' ),
            'dashicons-edit',
            27
        );
    }

    public function render_admin_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        if ( isset( $_POST['abcgs_start'] ) ) {
            $this->handle_start_queue();
        }

        if ( isset( $_POST['abcgs_analyze_site'] ) ) {
            $this->handle_analyze_site();
        }

        $settings = get_option( self::OPTION_SETTINGS, array(
            'api_key'         => '',
            'model'           => 'gpt-4.1-mini',
            'category_id'     => 0,
            'keyword_density' => 5,
            'post_status'     => 'publish',
            'post_length'     => 1200,
            'language'        => 'vi',
            'tone'            => 'neutral',
            'article_style'   => 'general_guide',
            'site_profile'    => '',
        ) );

        $api_key         = $settings['api_key'] ?? '';
        $model           = $settings['model'] ?? 'gpt-4.1-mini';
        $category_id     = intval( $settings['category_id'] ?? 0 );
        $keyword_density = floatval( $settings['keyword_density'] ?? 5 );
        $post_status     = $settings['post_status'] ?? 'publish';
        $post_length     = intval( $settings['post_length'] ?? 1200 );
        $language        = $settings['language'] ?? 'vi';
        $tone            = $settings['tone'] ?? 'neutral';
        $article_style   = $settings['article_style'] ?? 'general_guide';
        $site_profile    = $settings['site_profile'] ?? '';

        $available_models = array(
            'gpt-4.1-mini'  => 'gpt-4.1-mini (nhanh, tiết kiệm)',
            'gpt-4.1'       => 'gpt-4.1 (chất lượng cao)',
            'gpt-4o-mini'   => 'gpt-4o-mini (tối ưu chi phí)',
            'gpt-4o'        => 'gpt-4o (đa năng cao cấp)',
            'o3-mini'       => 'o3-mini (lý luận mạnh)',
            'gpt-3.5-turbo' => 'gpt-3.5-turbo (đời cũ, giá rẻ)',
        );

        $model_is_known = array_key_exists( $model, $available_models );
        $custom_model   = $model_is_known ? '' : $model;

        $queue      = get_option( self::OPTION_QUEUE, array() );
        $queue_size = is_array( $queue ) ? count( $queue ) : 0;

        $silo_log = get_option( self::OPTION_SILO_LOG, '' );

        $license = get_option( self::OPTION_LICENSE, array(
            'serial'     => '',
            'email'      => '',
            'status'     => 'invalid',
            'max_posts'  => 0,
            'used_posts' => 0,
            'plan'       => '',
        ) );

        $max_posts  = intval( $license['max_posts'] ?? 0 );
        $used_posts = intval( $license['used_posts'] ?? 0 );
        $remain     = max( 0, $max_posts - $used_posts );
        ?>

        <div class="wrap">
            <h1>AI Bulk ChatGPT Silo</h1>

            <?php if ( isset( $_GET['abcgs_message'] ) ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php echo esc_html( wp_unslash( $_GET['abcgs_message'] ) ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( isset( $_GET['abcgs_error'] ) ) : ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php echo esc_html( wp_unslash( $_GET['abcgs_error'] ) ); ?></p>
                </div>
            <?php endif; ?>

            <form method="post">
                <?php wp_nonce_field( 'abcgs_start_queue', 'abcgs_nonce' ); ?>

                <h2>License / Serial key</h2>
                <table class="form-table">
                    <tr>
                        <th><label for="abcgs_serial_key">Seri key</label></th>
                        <td>
                            <input type="text"
                                   name="abcgs_serial_key"
                                   id="abcgs_serial_key"
                                   value="<?php echo esc_attr( $license['serial'] ?? '' ); ?>"
                                   class="regular-text"
                                   placeholder="VD: FM-XXXX-XXXX-XXXX">
                            <br><br>
                            <label for="abcgs_license_email"><strong>Email license</strong></label><br>
                            <input type="email"
                                   name="abcgs_license_email"
                                   id="abcgs_license_email"
                                   value="<?php echo esc_attr( $license['email'] ?? '' ); ?>"
                                   class="regular-text"
                                   placeholder="Email dùng khi mua plugin">

                            <?php
                            $status_text  = 'Chưa kích hoạt';
                            $status_class = 'notice-warning';
                            if ( isset( $license['status'] ) && $license['status'] === 'valid' ) {
                                $status_text  = 'License hợp lệ';
                                $status_class = 'notice-success';
                            } elseif ( isset( $license['status'] ) && $license['status'] === 'invalid' && ! empty( $license['serial'] ) ) {
                                $status_text  = 'License không hợp lệ hoặc đã hết hạn';
                                $status_class = 'notice-error';
                            }
                            ?>
                            <div class="notice <?php echo esc_attr( $status_class ); ?>" style="margin-top:10px;padding:8px;">
                                <p>
                                    Trạng thái license: <strong><?php echo esc_html( $status_text ); ?></strong><br>
                                    Gói: <strong><?php echo esc_html( $license['plan'] ?? '' ); ?></strong><br>
                                    Email license: <strong><?php echo esc_html( $license['email'] ?? '' ); ?></strong><br>
                                    Số bài đã tạo (server): <strong><?php echo esc_html( $used_posts ); ?></strong><br>
                                    Giới hạn tối đa: <strong><?php echo esc_html( $max_posts ); ?></strong><br>
                                    Còn lại (server): <strong><?php echo esc_html( $remain ); ?></strong>
                                </p>
                            </div>
                            <p class="description">
                                License gắn với 1 email + domain + giới hạn số bài (30 / 50 / 80 / 500 / 800 / 1000).
                            </p>
                        </td>
                    </tr>
                </table>

                <h2>Cấu hình API</h2>
                <table class="form-table">
                    <tr>
                        <th><label for="abcgs_api_key">OpenAI API key</label></th>
                        <td>
                            <input type="password"
                                   name="abcgs_api_key"
                                   id="abcgs_api_key"
                                   value="<?php echo esc_attr( $api_key ); ?>"
                                   class="regular-text"
                                   placeholder="sk-...">
                            <p class="description">Dán API key OpenAI.</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="abcgs_model_select">Model ChatGPT</label></th>
                        <td>
                            <select name="abcgs_model_select" id="abcgs_model_select">
                                <?php foreach ( $available_models as $value => $label ) : ?>
                                    <option value="<?php echo esc_attr( $value ); ?>" <?php selected( $model_is_known ? $model : '', $value ); ?>>
                                        <?php echo esc_html( $label ); ?>
                                    </option>
                                <?php endforeach; ?>
                                <option value="custom" <?php selected( ! $model_is_known ); ?>>Model khác (nhập tay)</option>
                            </select>
                            <p style="margin-top:8px;">
                                <label for="abcgs_model_custom"><strong>Model tùy chỉnh:</strong></label><br>
                                <input type="text"
                                       name="abcgs_model_custom"
                                       id="abcgs_model_custom"
                                       value="<?php echo esc_attr( $custom_model ); ?>"
                                       class="regular-text"
                                       placeholder="VD: gpt-4.1-mini">
                            </p>
                            <p class="description">Nếu chọn "Model khác" thì sẽ dùng model trong ô này.</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="abcgs_language">Ngôn ngữ bài viết</label></th>
                        <td>
                            <select name="abcgs_language" id="abcgs_language">
                                <option value="vi" <?php selected( $language, 'vi' ); ?>>Tiếng Việt</option>
                                <option value="en" <?php selected( $language, 'en' ); ?>>Tiếng Anh</option>
                                <option value="zh" <?php selected( $language, 'zh' ); ?>>Tiếng Trung (giản thể)</option>
                            </select>
                            <p class="description">Chọn ngôn ngữ bài viết được sinh ra.</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="abcgs_category_id">Chuyên mục</label></th>
                        <td>
                            <?php
                            wp_dropdown_categories( array(
                                'show_option_all' => '— Chọn chuyên mục —',
                                'hide_empty'      => false,
                                'name'            => 'abcgs_category_id',
                                'id'              => 'abcgs_category_id',
                                'selected'        => $category_id,
                                'taxonomy'        => 'category',
                            ) );
                            ?>
                            <p class="description">Tất cả bài viết sẽ đăng vào chuyên mục này.</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="abcgs_post_status">Trạng thái bài viết</label></th>
                        <td>
                            <select name="abcgs_post_status" id="abcgs_post_status">
                                <option value="publish" <?php selected( $post_status, 'publish' ); ?>>Đăng ngay (publish)</option>
                                <option value="draft" <?php selected( $post_status, 'draft' ); ?>>Lưu nháp (draft)</option>
                            </select>
                        </td>
                    </tr>
                </table>

                <h2>Tùy chọn nội dung</h2>
                <table class="form-table">
                    <tr>
                        <th><label for="abcgs_post_length">Độ dài bài viết (số từ)</label></th>
                        <td>
                            <input type="number"
                                   name="abcgs_post_length"
                                   id="abcgs_post_length"
                                   value="<?php echo esc_attr( $post_length ); ?>"
                                   min="200"
                                   step="100">
                            <p class="description">VD: 800, 1200, 2000...</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="abcgs_keyword_density">Mật độ từ khóa (%)</label></th>
                        <td>
                            <input type="number"
                                   name="abcgs_keyword_density"
                                   id="abcgs_keyword_density"
                                   value="<?php echo esc_attr( $keyword_density ); ?>"
                                   min="1"
                                   max="20"
                                   step="0.5">
                            <p class="description">VD: 3, 3.5, 5 (%).</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="abcgs_tone">Tone bài viết</label></th>
                        <td>
                            <select name="abcgs_tone" id="abcgs_tone">
                                <option value="neutral" <?php selected( $tone, 'neutral' ); ?>>Trung lập</option>
                                <option value="friendly" <?php selected( $tone, 'friendly' ); ?>>Thân thiện</option>
                                <option value="formal" <?php selected( $tone, 'formal' ); ?>>Trang trọng</option>
                                <option value="expert" <?php selected( $tone, 'expert' ); ?>>Chuyên gia</option>
                                <option value="promotional" <?php selected( $tone, 'promotional' ); ?>>Quảng cáo nhẹ</option>
                                <option value="journalistic" <?php selected( $tone, 'journalistic' ); ?>>Báo chí</option>
                                <option value="storytelling" <?php selected( $tone, 'storytelling' ); ?>>Kể chuyện</option>
                            </select>
                            <p class="description">Chọn giọng viết chính cho toàn bộ bài.</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="abcgs_article_style">Kiểu bài viết</label></th>
                        <td>
                            <select name="abcgs_article_style" id="abcgs_article_style">
                                <option value="general_guide" <?php selected( $article_style, 'general_guide' ); ?>>Giới thiệu / hướng dẫn tổng quan</option>
                                <option value="news_article" <?php selected( $article_style, 'news_article' ); ?>>Tin tức / cập nhật</option>
                                <option value="product_description" <?php selected( $article_style, 'product_description' ); ?>>Mô tả sản phẩm</option>
                                <option value="product_review" <?php selected( $article_style, 'product_review' ); ?>>Đánh giá sản phẩm</option>
                                <option value="comparison" <?php selected( $article_style, 'comparison' ); ?>>So sánh</option>
                                <option value="how_to" <?php selected( $article_style, 'how_to' ); ?>>How-to</option>
                                <option value="listicle" <?php selected( $article_style, 'listicle' ); ?>>Bài danh sách (Top X)</option>
                            </select>
                            <p class="description">Giúp AI viết đúng định dạng bài.</p>
                        </td>
                    </tr>
                </table>

                <h2>Hồ sơ lĩnh vực website (context cho AI)</h2>
                <p><em>Plugin sẽ dựa vào hồ sơ này để viết đúng lĩnh vực website. Có thể tự phân tích hoặc chỉnh sửa bằng tay.</em></p>
                <textarea name="abcgs_site_profile" rows="5" style="width:100%;max-width:800px;"><?php echo esc_textarea( $site_profile ); ?></textarea>
                <p class="description">
                    Mô tả nhanh website tập trung vào lĩnh vực gì (VD: nhà hàng, du lịch, vay online...).<br>
                    Nếu muốn plugin tự phân tích các bài đã có, bấm nút bên dưới.
                </p>
                <p>
                    <button type="submit" name="abcgs_analyze_site" class="button">Tự động phân tích bài viết trên website</button>
                </p>

                <hr>

                <h2>Danh sách từ khóa</h2>
                <p>
                    Mỗi dòng là <strong>một từ khóa</strong>.<br>
                    Dòng 1 = <strong>silo page</strong> (bài trụ cột).<br>
                    Các dòng sau = bài silo con.<br>
                    Plugin sẽ tự tạo <strong>link nội bộ silo</strong> khi hoàn thành.
                </p>
                <textarea name="abcgs_keywords" rows="10" style="width:100%;max-width:800px;" placeholder="Ví dụ:
Đào Tạo SEO Fame Media
Khóa Học SEO Fame Media
Đào Tạo SEO TpHCM Fame Media
Khóa học SEO online Fame Media"></textarea>

                <p class="submit">
                    <button type="submit" name="abcgs_start" class="button button-primary button-large">
                        Tạo và đăng bài viết ngay (Queue nền)
                    </button>
                </p>
            </form>

            <hr>

            <h2>Trạng thái queue</h2>
            <p>
                Số mục trong queue: <strong><?php echo intval( $queue_size ); ?></strong><br>
                <?php
                $pending = 0;
                $done    = 0;
                if ( $queue_size > 0 ) {
                    foreach ( $queue as $item ) {
                        if ( isset( $item['status'] ) && $item['status'] === 'done' ) {
                            $done++;
                        } else {
                            $pending++;
                        }
                    }
                }
                ?>
                Còn đang xử lý: <strong><?php echo intval( $pending ); ?></strong><br>
                Đã xong: <strong><?php echo intval( $done ); ?></strong>
            </p>

            <?php if ( ! empty( $silo_log ) ) : ?>
                <p><strong>Trạng thái silo:</strong> <?php echo esc_html( $silo_log ); ?></p>
            <?php endif; ?>

            <?php if ( $queue_size > 0 ) : ?>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Từ khóa</th>
                            <th>ID bài viết</th>
                            <th>Trạng thái</th>
                            <th>Ghi chú</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php
                    $i = 1;
                    foreach ( $queue as $item ) :
                        ?>
                        <tr>
                            <td><?php echo intval( $i ); ?></td>
                            <td><?php echo esc_html( $item['keyword'] ?? '' ); ?></td>
                            <td><?php echo ! empty( $item['post_id'] ) ? intval( $item['post_id'] ) : '-'; ?></td>
                            <td><?php echo esc_html( $item['status'] ?? '' ); ?></td>
                            <td><?php echo esc_html( $item['content_status'] ?? '' ); ?></td>
                        </tr>
                        <?php
                        $i++;
                    endforeach;
                    ?>
                    </tbody>
                </table>
            <?php else : ?>
                <p>Chưa có queue nào hoặc đã xử lý xong.</p>
            <?php endif; ?>
        </div>

        <?php
    }

    private function handle_start_queue() {
        if ( ! isset( $_POST['abcgs_nonce'] ) || ! wp_verify_nonce( $_POST['abcgs_nonce'], 'abcgs_start_queue' ) ) {
            wp_die( 'Xác thực không hợp lệ.' );
        }

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'Không có quyền.' );
        }

        $api_key          = isset( $_POST['abcgs_api_key'] ) ? sanitize_text_field( $_POST['abcgs_api_key'] ) : '';
        $model_select     = isset( $_POST['abcgs_model_select'] ) ? sanitize_text_field( $_POST['abcgs_model_select'] ) : 'gpt-4.1-mini';
        $model_custom_raw = isset( $_POST['abcgs_model_custom'] ) ? sanitize_text_field( $_POST['abcgs_model_custom'] ) : '';
        $category_id      = isset( $_POST['abcgs_category_id'] ) ? intval( $_POST['abcgs_category_id'] ) : 0;
        $post_status      = isset( $_POST['abcgs_post_status'] ) ? sanitize_text_field( $_POST['abcgs_post_status'] ) : 'publish';
        $post_length      = isset( $_POST['abcgs_post_length'] ) ? intval( $_POST['abcgs_post_length'] ) : 1200;
        $keyword_density  = isset( $_POST['abcgs_keyword_density'] ) ? floatval( $_POST['abcgs_keyword_density'] ) : 5;
        $language         = isset( $_POST['abcgs_language'] ) ? sanitize_text_field( $_POST['abcgs_language'] ) : 'vi';
        $tone             = isset( $_POST['abcgs_tone'] ) ? sanitize_text_field( $_POST['abcgs_tone'] ) : 'neutral';
        $article_style    = isset( $_POST['abcgs_article_style'] ) ? sanitize_text_field( $_POST['abcgs_article_style'] ) : 'general_guide';
        $site_profile     = isset( $_POST['abcgs_site_profile'] ) ? wp_kses_post( wp_unslash( $_POST['abcgs_site_profile'] ) ) : '';

        $serial_key       = isset( $_POST['abcgs_serial_key'] ) ? sanitize_text_field( $_POST['abcgs_serial_key'] ) : '';
        $license_email    = isset( $_POST['abcgs_license_email'] ) ? sanitize_email( $_POST['abcgs_license_email'] ) : '';

        if ( $model_select === 'custom' ) {
            $model = ! empty( $model_custom_raw ) ? $model_custom_raw : 'gpt-4.1-mini';
        } else {
            $model = $model_select;
        }

        if ( empty( $api_key ) || strlen( $api_key ) < 20 ) {
            $this->redirect_with_error( 'API key trống hoặc không hợp lệ.' );
        }

        if ( $category_id <= 0 ) {
            $this->redirect_with_error( 'Vui lòng chọn chuyên mục.' );
        }

        if ( $post_length < 200 ) {
            $post_length = 200;
        }

        if ( $keyword_density <= 0 ) {
            $keyword_density = 3;
        }

        if ( ! in_array( $post_status, array( 'publish', 'draft' ), true ) ) {
            $post_status = 'publish';
        }

        if ( ! in_array( $language, array( 'vi', 'en', 'zh' ), true ) ) {
            $language = 'vi';
        }

        if ( empty( $_POST['abcgs_keywords'] ) ) {
            $this->redirect_with_error( 'Vui lòng nhập danh sách từ khóa.' );
        }

        if ( empty( $serial_key ) ) {
            $this->redirect_with_error( 'Vui lòng nhập seri key.' );
        }

        if ( empty( $license_email ) ) {
            $this->redirect_with_error( 'Vui lòng nhập email license.' );
        }

        $raw_keywords = wp_unslash( $_POST['abcgs_keywords'] );
        $lines        = preg_split( '/\r\n|\r|\n/', $raw_keywords );
        $keywords     = array();

        foreach ( $lines as $line ) {
            $line = trim( $line );
            if ( ! empty( $line ) ) {
                $keywords[] = $line;
            }
        }

        if ( empty( $keywords ) ) {
            $this->redirect_with_error( 'Danh sách từ khóa không hợp lệ.' );
        }

        $license_info  = array();
        $license_error = '';

        if ( ! $this->validate_license_key( $serial_key, $license_email, $license_info, $license_error ) ) {
            update_option( self::OPTION_LICENSE, array(
                'serial'     => $serial_key,
                'email'      => $license_email,
                'status'     => 'invalid',
                'max_posts'  => 0,
                'used_posts' => 0,
                'plan'       => '',
            ) );
            $this->redirect_with_error( 'License không hợp lệ: ' . $license_error );
        }

        $used_posts = intval( $license_info['used_posts'] ?? 0 );
        $max_posts  = intval( $license_info['max_posts'] ?? 0 );
        $plan       = sanitize_text_field( $license_info['plan'] ?? '' );

        if ( $max_posts <= 0 ) {
            $this->redirect_with_error( 'License hợp lệ nhưng max_posts <= 0.' );
        }

        if ( $used_posts >= $max_posts ) {
            update_option( self::OPTION_LICENSE, array(
                'serial'     => $serial_key,
                'email'      => $license_email,
                'status'     => 'valid',
                'max_posts'  => $max_posts,
                'used_posts' => $used_posts,
                'plan'       => $plan,
            ) );
            $this->redirect_with_error( 'Bạn đã dùng hết số bài cho license này.' );
        }

        update_option( self::OPTION_LICENSE, array(
            'serial'     => $serial_key,
            'email'      => $license_email,
            'status'     => 'valid',
            'max_posts'  => $max_posts,
            'used_posts' => $used_posts,
            'plan'       => $plan,
        ) );

        update_option( self::OPTION_SETTINGS, array(
            'api_key'         => $api_key,
            'model'           => $model,
            'category_id'     => $category_id,
            'keyword_density' => $keyword_density,
            'post_status'     => $post_status,
            'post_length'     => $post_length,
            'language'        => $language,
            'tone'            => $tone,
            'article_style'   => $article_style,
            'site_profile'    => $site_profile,
        ) );

        $queue = array();
        foreach ( $keywords as $kw ) {
            $queue[] = array(
                'keyword'        => $kw,
                'post_id'        => 0,
                'status'         => 'pending',
                'content_status' => '',
            );
        }

        update_option( self::OPTION_QUEUE, $queue );
        update_option( self::OPTION_SILO_LOG, '' );

        if ( ! wp_next_scheduled( self::CRON_HOOK ) ) {
            wp_schedule_single_event( time() + 10, self::CRON_HOOK );
        }

        $this->redirect_with_message( 'Đã xếp hàng ' . count( $queue ) . ' từ khóa. Cron sẽ lần lượt tạo bài và link silo.' );
    }

    private function handle_analyze_site() {
        if ( ! isset( $_POST['abcgs_nonce'] ) || ! wp_verify_nonce( $_POST['abcgs_nonce'], 'abcgs_start_queue' ) ) {
            wp_die( 'Xác thực không hợp lệ.' );
        }

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'Không có quyền.' );
        }

        $settings = get_option( self::OPTION_SETTINGS, array() );
        $site_profile = $this->build_site_profile_summary();

        $settings['site_profile'] = $site_profile;
        update_option( self::OPTION_SETTINGS, $settings );

        $this->redirect_with_message( 'Đã phân tích website và cập nhật hồ sơ lĩnh vực website.' );
    }

    private function build_site_profile_summary() {
        $args = array(
            'post_type'      => 'post',
            'post_status'    => 'publish',
            'posts_per_page' => 30,
            'orderby'        => 'date',
            'order'          => 'DESC',
            'fields'         => 'ids',
        );

        $posts = get_posts( $args );
        if ( empty( $posts ) ) {
            return 'Website chưa có bài viết hoặc dữ liệu ít. Chủ đề chính: chưa xác định rõ.';
        }

        $cat_names = array();
        $tag_names = array();

        foreach ( $posts as $pid ) {
            $cats = get_the_category( $pid );
            if ( $cats && ! is_wp_error( $cats ) ) {
                foreach ( $cats as $c ) {
                    $cat_names[] = $c->name;
                }
            }

            $tags = get_the_terms( $pid, 'post_tag' );
            if ( $tags && ! is_wp_error( $tags ) ) {
                foreach ( $tags as $t ) {
                    $tag_names[] = $t->name;
                }
            }
        }

        $top_cats = array();
        if ( ! empty( $cat_names ) ) {
            $counts = array_count_values( $cat_names );
            arsort( $counts );
            $top_cats = array_slice( array_keys( $counts ), 0, 8 );
        }

        $top_tags = array();
        if ( ! empty( $tag_names ) ) {
            $counts = array_count_values( $tag_names );
            arsort( $counts );
            $top_tags = array_slice( array_keys( $counts ), 0, 12 );
        }

        $cat_part  = ! empty( $top_cats ) ? 'Website tập trung vào các chuyên mục: ' . implode( ', ', $top_cats ) . '.' : 'Không xác định rõ chuyên mục chính.';
        $tag_part  = ! empty( $top_tags ) ? ' Từ khóa nổi bật thường gặp: ' . implode( ', ', $top_tags ) . '.' : '';
        $extra     = ' Sử dụng thông tin này làm context để viết bài đúng lĩnh vực của website.';

        return $cat_part . $tag_part . $extra;
    }

    public function process_queue() {
        $queue = get_option( self::OPTION_QUEUE, array() );
        if ( empty( $queue ) || ! is_array( $queue ) ) {
            return;
        }

        $settings = get_option( self::OPTION_SETTINGS, array() );
        $api_key  = $settings['api_key'] ?? '';
        $model    = $settings['model'] ?? 'gpt-4.1-mini';

        if ( empty( $api_key ) || strlen( $api_key ) < 20 ) {
            return;
        }

        $category_id     = intval( $settings['category_id'] ?? 0 );
        $post_status     = $settings['post_status'] ?? 'publish';
        $post_length     = intval( $settings['post_length'] ?? 1200 );
        $keyword_density = floatval( $settings['keyword_density'] ?? 5 );
        $language        = $settings['language'] ?? 'vi';
        $tone            = $settings['tone'] ?? 'neutral';
        $article_style   = $settings['article_style'] ?? 'general_guide';
        $site_profile    = $settings['site_profile'] ?? '';

        $license = get_option( self::OPTION_LICENSE, array(
            'serial'     => '',
            'email'      => '',
            'status'     => 'invalid',
            'max_posts'  => 0,
            'used_posts' => 0,
            'plan'       => '',
        ) );

        $serial_valid = ( isset( $license['status'] ) && $license['status'] === 'valid' );
        $max_posts    = intval( $license['max_posts'] ?? 0 );
        $used_posts   = intval( $license['used_posts'] ?? 0 );

        $batch_size = 2;
        $processed  = 0;

        foreach ( $queue as $index => $item ) {
            if ( $processed >= $batch_size ) {
                break;
            }

            if ( isset( $item['status'] ) && $item['status'] === 'done' ) {
                continue;
            }

            $keyword = $item['keyword'] ?? '';
            if ( empty( $keyword ) ) {
                $queue[ $index ]['status']         = 'done';
                $queue[ $index ]['content_status'] = 'Bỏ qua: từ khóa rỗng.';
                continue;
            }

            if ( ! $serial_valid ) {
                $queue[ $index ]['status']         = 'done';
                $queue[ $index ]['content_status'] = 'Dừng: license không hợp lệ hoặc chưa kích hoạt.';
                continue;
            }

            if ( $used_posts >= $max_posts ) {
                $queue[ $index ]['status']         = 'done';
                $queue[ $index ]['content_status'] = 'Dừng: đã hết số bài theo license.';
                continue;
            }

            if ( empty( $item['post_id'] ) ) {

                $prompt  = $this->build_prompt_for_keyword(
                    $keyword,
                    $post_length,
                    $keyword_density,
                    $language,
                    $tone,
                    $article_style,
                    $site_profile
                );

                $error_message_content = '';
                $content = $this->call_chatgpt( $api_key, $model, $prompt, $error_message_content );

                if ( ! $content ) {
                    $queue[ $index ]['content_status'] = 'Lỗi API: ' . $error_message_content;
                    $queue[ $index ]['status']         = 'done';
                    $processed++;
                    continue;
                }

                // ✅ Làm sạch: bỏ ```html ``` và bỏ H1 (nếu AI lỡ tạo)
                $content = $this->strip_code_fences( $content );
                $content = $this->remove_h1_tags( $content );

                $post_title = $this->extract_title_from_content( $content, $keyword );
                $excerpt    = $this->generate_excerpt_from_html( $content, 40 );

                $post_id = wp_insert_post( array(
                    'post_title'    => $post_title,
                    'post_content'  => $content,
                    'post_status'   => in_array( $post_status, array( 'publish', 'draft' ), true ) ? $post_status : 'publish',
                    'post_author'   => get_current_user_id(),
                    'post_category' => $category_id > 0 ? array( $category_id ) : array(),
                    'post_excerpt'  => $excerpt,
                ), true );

                if ( is_wp_error( $post_id ) || ! $post_id ) {
                    $queue[ $index ]['content_status'] = 'Lỗi: không tạo được bài viết.';
                    $queue[ $index ]['status']         = 'done';
                    $processed++;
                    continue;
                }

                // ✅ Trừ quota ở server (consume)
                $license = $this->consume_remote_license_usage( $license, 1 );
                $used_posts = intval( $license['used_posts'] ?? $used_posts );
                $max_posts  = intval( $license['max_posts']  ?? $max_posts );

                update_option( self::OPTION_LICENSE, array(
                    'serial'     => $license['serial'] ?? '',
                    'email'      => $license['email'] ?? '',
                    'status'     => 'valid',
                    'max_posts'  => $max_posts,
                    'used_posts' => $used_posts,
                    'plan'       => $license['plan'] ?? ( $license['plan'] ?? '' ),
                ) );

                $queue[ $index ]['post_id']        = $post_id;
                $queue[ $index ]['content_status'] = 'OK: đã tạo bài.';
            }

            $queue[ $index ]['status'] = 'done';
            $processed++;
        }

        update_option( self::OPTION_QUEUE, $queue );

        $pending_left = false;
        foreach ( $queue as $it ) {
            if ( ( $it['status'] ?? '' ) !== 'done' ) {
                $pending_left = true;
                break;
            }
        }

        if ( $pending_left ) {
            wp_schedule_single_event( time() + 30, self::CRON_HOOK );
        } else {
            $post_ids = array();
            $keywords = array();
            foreach ( $queue as $it ) {
                if ( ! empty( $it['post_id'] ) ) {
                    $post_ids[] = intval( $it['post_id'] );
                    $keywords[] = $it['keyword'] ?? '';
                }
            }

            if ( count( $post_ids ) >= 2 ) {
                $silo_error = '';
                $silo_ok    = $this->apply_silo_internal_links( $post_ids, $keywords, $silo_error );
                if ( $silo_ok ) {
                    update_option( self::OPTION_SILO_LOG, 'OK: đã gắn link silo cho ' . count( $post_ids ) . ' bài.' );
                } else {
                    update_option( self::OPTION_SILO_LOG, 'Lỗi khi gắn silo: ' . $silo_error );
                }
            } else {
                update_option( self::OPTION_SILO_LOG, 'Không đủ bài để tạo silo (tối thiểu 2 bài).' );
            }
        }
    }

    private function build_prompt_for_keyword( $keyword, $post_length, $density, $language, $tone, $article_style, $site_profile ) {
        $post_length     = max( 200, intval( $post_length ) );
        $keyword_density = floatval( $density );

        $tone_text = '';
        switch ( $tone ) {
            case 'friendly':
                $tone_text = 'Thân thiện, gần gũi, dễ hiểu, nhưng vẫn chuyên nghiệp.';
                break;
            case 'formal':
                $tone_text = 'Trang trọng, chuyên nghiệp, hạn chế tiếng lóng.';
                break;
            case 'expert':
                $tone_text = 'Chuyên gia, phân tích sâu, có góc nhìn chuyên môn nhưng vẫn dễ hiểu.';
                break;
            case 'promotional':
                $tone_text = 'Quảng cáo nhẹ, nhấn mạnh lợi ích, thúc đẩy hành động, nhưng không hứa hẹn/cam kết.';
                break;
            case 'journalistic':
                $tone_text = 'Báo chí, khách quan, cấu trúc rõ ràng như bài tin.';
                break;
            case 'storytelling':
                $tone_text = 'Kể chuyện theo dòng tự nhiên, nhưng không kể chuyện cá nhân cụ thể.';
                break;
            default:
                $tone_text = 'Trung lập, dễ hiểu, chuyên nghiệp.';
                break;
        }

        $style_text = '';
        switch ( $article_style ) {
            case 'news_article':
                $style_text = 'Phong cách tin tức/cập nhật: mở bài, nội dung chính, kết luận.';
                break;
            case 'product_description':
                $style_text = 'Mô tả sản phẩm/dịch vụ: tính năng, lợi ích, đối tượng phù hợp. Không nêu giá, không cam kết.';
                break;
            case 'product_review':
                $style_text = 'Đánh giá/review: ưu điểm, hạn chế, khách quan, không cam kết.';
                break;
            case 'comparison':
                $style_text = 'So sánh: tiêu chí rõ ràng, điểm giống/khác, kết luận theo nhu cầu.';
                break;
            case 'how_to':
                $style_text = 'How-to: các bước rõ ràng, có tiêu đề con cho từng bước.';
                break;
            case 'listicle':
                $style_text = 'Listicle: dạng Top X, mỗi mục có giải thích ngắn gọn.';
                break;
            default:
                $style_text = 'Giới thiệu tổng quan: dạng blog chia sẻ kiến thức/kinh nghiệm.';
                break;
        }

        $context_part = '';
        if ( ! empty( $site_profile ) ) {
            $context_part = "Context về website (lĩnh vực, chủ đề chính): " . $site_profile . "\nHãy giữ nội dung phù hợp với lĩnh vực và tệp khách hàng của website này.\n";
        }

        // ✅ Yêu cầu bỏ thẻ H1: title sẽ là dòng đầu tiên dạng văn bản, plugin tự đặt post_title
        if ( $language === 'vi' ) {
            return "
Bạn là một copywriter SEO chuyên nghiệp. Hãy viết một bài viết bằng tiếng Việt về từ khóa chính \"{$keyword}\" theo các yêu cầu sau:

{$context_part}
Giọng viết (tone): {$tone_text}
Kiểu bài viết: {$style_text}

1. Tiêu đề:
- Tự tạo một tiêu đề chính KHÔNG trùng hoàn toàn với từ khóa chính.
- Tiêu đề có CHÍNH XÁC 12 từ (đếm theo khoảng trắng).

2. Độ dài:
- Nội dung có độ dài là {$post_length} từ (có thể chênh lệch nhẹ).

3. Quy định bắt buộc:
- KHÔNG dùng dấu gạch ngang '-' trong toàn bộ bài.
- KHÔNG nêu giá sản phẩm.
- KHÔNG kể câu chuyện cá nhân cụ thể.
- KHÔNG đưa ra bất kỳ cam kết hay đảm bảo nào.
- KHÔNG có mục Câu hỏi thường gặp (FAQ).

4. Từ khóa:
- Từ khóa chính: \"{$keyword}\".
- Mật độ từ khóa chính khoảng {$keyword_density}% trên toàn bài (ước lượng hợp lý).
- Dùng thêm từ khóa phụ, từ khóa liên quan, semantic keyword và từ khóa ngữ cảnh liên quan tới từ khóa chính.
- Mỗi lần xuất hiện từ khóa chính \"{$keyword}\", in đậm bằng HTML: <strong>{$keyword}</strong>.

5. Định dạng:
- Viết tiếng Việt có dấu, câu từ tự nhiên, dễ đọc.
- Chỉ dùng các thẻ HTML cơ bản: <h2>, <h3>, <p>, <ul>, <li>.
- KHÔNG dùng <h1>.
- KHÔNG chèn script, iframe hay mã nhúng.

Trả về TOÀN BỘ nội dung bài viết dưới dạng HTML (không kèm ``` hoặc ```html):
";
        }

        if ( $language === 'zh' ) {
            return "
You are an SEO copywriter. Write an informational article in Simplified Chinese about the main keyword \"{$keyword}\" with the following rules:

{$context_part}
Tone of voice: {$tone_text}
Article style: {$style_text}

1. Title:
- Create a unique main title that is NOT exactly the same as the main keyword.
- The title must contain EXACTLY 12 segments separated by spaces.

2. Length:
- The article length should be around {$post_length} words (approximate is fine).

3. Strict rules:
- Do NOT use hyphen '-' in the article.
- Do NOT mention any product price.
- Do NOT tell personal stories.
- Do NOT make any promises or guarantees.
- Do NOT include any FAQ section.

4. Keywords:
- Main keyword: \"{$keyword}\".
- Keep the main keyword density around {$keyword_density}% (roughly).
- Use related keywords, semantic keywords, and latent keywords related to the main keyword.
- Every time the main keyword \"{$keyword}\" appears, wrap it with <strong>{$keyword}</strong>.

5. Formatting:
- Use only <h2>, <h3>, <p>, <ul>, <li>.
- Do NOT use <h1>.
- No scripts, no iframes, no embeds.
- Return FULL HTML only, no code fences.

Return FULL HTML content only:
";
        }

        return "
You are an SEO copywriter. Write an article in English about the main keyword \"{$keyword}\" with the following rules:

{$context_part}
Tone of voice: {$tone_text}
Article style: {$style_text}

1. Title:
- Create a unique main title that is NOT exactly the same as the main keyword.
- The title must contain EXACTLY 12 words.

2. Length:
- The article length should be around {$post_length} words (approximate is fine).

3. Strict rules:
- Do NOT use hyphen '-' in the article.
- Do NOT mention any product price.
- Do NOT tell personal stories.
- Do NOT make any promises or guarantees.
- Do NOT include any FAQ section.

4. Keywords:
- Main keyword: \"{$keyword}\".
- Keep the main keyword density around {$keyword_density}% (roughly).
- Use related keywords, semantic keywords, and ghost keywords related to the main keyword.
- Every time the main keyword \"{$keyword}\" appears, wrap it with <strong>{$keyword}</strong>.

5. Formatting:
- Use only <h2>, <h3>, <p>, <ul>, <li>.
- Do NOT use <h1>.
- No scripts, no iframes, no embeds.
- Return FULL HTML only, no code fences.

Return FULL HTML content only:
";
    }

    private function call_chatgpt( $api_key, $model, $prompt, &$error_message = '' ) {
        $error_message = '';

        $response = wp_remote_post(
            'https://api.openai.com/v1/chat/completions',
            array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api_key,
                    'Content-Type'  => 'application/json',
                ),
                'body'    => wp_json_encode( array(
                    'model'    => $model,
                    'messages' => array(
                        array(
                            'role'    => 'system',
                            'content' => 'You are a strict SEO content writer. Follow instructions exactly. Return only HTML without code fences.',
                        ),
                        array(
                            'role'    => 'user',
                            'content' => $prompt,
                        ),
                    ),
                    'temperature' => 0.7,
                ) ),
                'timeout' => 60,
            )
        );

        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            return false;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        if ( $code < 200 || $code >= 300 ) {
            $error_message = 'HTTP ' . $code . ' | ' . substr( $body, 0, 300 );
            return false;
        }

        $data = json_decode( $body, true );
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            $error_message = 'JSON error: ' . json_last_error_msg();
            return false;
        }

        if ( empty( $data['choices'][0]['message']['content'] ) ) {
            $error_message = 'API không trả về content.';
            return false;
        }

        return $data['choices'][0]['message']['content'];
    }

    private function strip_code_fences( $content ) {
        $content = trim( $content );
        // Bỏ ```...``` và ```html...```
        $content = preg_replace( '/^```[a-zA-Z0-9]*\s*/', '', $content );
        $content = preg_replace( '/\s*```$/', '', $content );
        return trim( $content );
    }

    private function remove_h1_tags( $content ) {
        // Nếu AI lỡ chèn <h1> thì chuyển thành <h2>
        $content = preg_replace( '/<\s*h1\b([^>]*)>/i', '<h2$1>', $content );
        $content = preg_replace( '/<\s*\/\s*h1\s*>/i', '</h2>', $content );
        return $content;
    }

    private function extract_title_from_content( $content, $fallback ) {
        // Ưu tiên: dòng đầu tiên (plain text) trước thẻ HTML
        $text = trim( $content );
        $text_no_tags = wp_strip_all_tags( $text );
        $lines = preg_split( '/\r\n|\r|\n/', $text_no_tags );
        if ( ! empty( $lines[0] ) ) {
            $maybe = trim( preg_replace( '/\s+/', ' ', $lines[0] ) );
            if ( mb_strlen( $maybe ) >= 10 ) {
                return $maybe;
            }
        }

        // Nếu không có, thử lấy H2 đầu tiên
        if ( preg_match( '/<h2[^>]*>(.*?)<\/h2>/is', $content, $m ) ) {
            $clean = wp_strip_all_tags( $m[1] );
            $clean = trim( preg_replace( '/\s+/', ' ', $clean ) );
            if ( ! empty( $clean ) ) {
                return $clean;
            }
        }

        return $fallback;
    }

    private function generate_excerpt_from_html( $html, $word_limit = 40 ) {
        $text = wp_strip_all_tags( $html );
        $text = trim( preg_replace( '/\s+/', ' ', $text ) );
        if ( empty( $text ) ) {
            return '';
        }
        $words = explode( ' ', $text );
        if ( count( $words ) <= $word_limit ) {
            return $text;
        }
        return implode( ' ', array_slice( $words, 0, $word_limit ) ) . '...';
    }

    private function apply_silo_internal_links( $post_ids, $keywords, &$error_message = '' ) {
        $error_message = '';

        if ( empty( $post_ids ) || count( $post_ids ) < 2 ) {
            $error_message = 'Không đủ bài để tạo silo.';
            return false;
        }

        $count_posts    = count( $post_ids );
        $count_keywords = count( $keywords );
        $count          = min( $count_posts, $count_keywords );

        $silo_post_id = $post_ids[0];
        $silo_url     = get_permalink( $silo_post_id );
        $silo_anchor  = $keywords[0] ?? get_the_title( $silo_post_id );

        if ( ! $silo_url ) {
            $error_message = 'Không lấy được permalink silo page.';
            return false;
        }

        for ( $i = 0; $i < $count; $i++ ) {
            $post_id = $post_ids[ $i ];
            $content = get_post_field( 'post_content', $post_id );
            if ( empty( $content ) ) {
                continue;
            }

            $links = array();

            if ( $i === 0 ) {
                if ( $count > 1 ) {
                    $next_post_id = $post_ids[1];
                    $next_url     = get_permalink( $next_post_id );
                    $next_anchor  = $keywords[1] ?? get_the_title( $next_post_id );
                    if ( $next_url ) {
                        $links[] = sprintf( '<a href="%s">%s</a>', esc_url( $next_url ), esc_html( $next_anchor ) );
                    }
                }
            } else {
                if ( $i + 1 < $count ) {
                    $next_post_id = $post_ids[ $i + 1 ];
                    $next_url     = get_permalink( $next_post_id );
                    $next_anchor  = $keywords[ $i + 1 ] ?? get_the_title( $next_post_id );
                    if ( $next_url ) {
                        $links[] = sprintf( '<a href="%s">%s</a>', esc_url( $next_url ), esc_html( $next_anchor ) );
                    }
                }
                $links[] = sprintf( '<a href="%s">%s</a>', esc_url( $silo_url ), esc_html( $silo_anchor ) );
            }

            if ( empty( $links ) ) {
                continue;
            }

            $links_html = '<div class="silo-links"><p><strong>Liên kết nội bộ cùng chủ đề:</strong> ' .
                implode( ' | ', $links ) .
                '</p></div>';

            $new_content = $content . "\n\n" . $links_html;

            $updated = wp_update_post( array(
                'ID'           => $post_id,
                'post_content' => $new_content,
            ), true );

            if ( is_wp_error( $updated ) ) {
                $error_message = 'Lỗi update bài ' . $post_id . ': ' . $updated->get_error_message();
            }
        }

        return true;
    }

    private function validate_license_key( $serial_key, $license_email, &$license_info, &$error_message = '' ) {
        $error_message = '';
        $license_info  = array();

        $serial_key    = trim( $serial_key );
        $license_email = trim( $license_email );

        if ( empty( $serial_key ) ) {
            $error_message = 'Seri key trống.';
            return false;
        }
        if ( empty( $license_email ) ) {
            $error_message = 'Email license trống.';
            return false;
        }

        $site_url = site_url();
$host = parse_url( $site_url, PHP_URL_HOST );
if ( ! $host ) {
    $host = $site_url;
}
// Chuẩn hoá domain: bỏ www, bỏ port, lowercase
$domain = strtolower( preg_replace( '/^www\./i', '', preg_replace( '/:\\d+$/', '', (string) $host ) ) );

$response = wp_remote_post(
            self::LICENSE_API_URL,
            array(
                'timeout' => 25,
                'body'    => array(
                    'action'   => 'validate',
                    'serial'   => $serial_key,
                    'email'    => $license_email,
                    'domain'   => $domain,
                    'site_url' => $site_url,
                ),
            )
        );

        if ( is_wp_error( $response ) ) {
            $error_message = 'Không kết nối được license server: ' . $response->get_error_message();
            return false;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        if ( $code < 200 || $code >= 300 ) {
            $error_message = 'License server HTTP ' . $code . ' | ' . substr( $body, 0, 200 );
            return false;
        }

        $data = json_decode( $body, true );
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            $error_message = 'Không parse được JSON: ' . json_last_error_msg();
            return false;
        }

        
if ( empty( $data['status'] ) || $data['status'] !== 'ok' ) {
    $code_text = ! empty( $data['code'] ) ? sanitize_text_field( $data['code'] ) : 'LICENSE_INVALID';
    $msg       = ! empty( $data['message'] ) ? sanitize_text_field( $data['message'] ) : 'License không hợp lệ.';

    // Thêm thông tin domain/email để debug nhanh (nếu server có trả về)
    $extra = '';
    if ( ! empty( $data['expected_domain'] ) || ! empty( $data['received_domain'] ) ) {
        $extra .= ' | expected_domain=' . (string) ( $data['expected_domain'] ?? '' ) . ' | received_domain=' . (string) ( $data['received_domain'] ?? '' );
    }
    if ( ! empty( $data['expected_email'] ) || ! empty( $data['received_email'] ) ) {
        $extra .= ' | expected_email=' . (string) ( $data['expected_email'] ?? '' ) . ' | received_email=' . (string) ( $data['received_email'] ?? '' );
    }

    $error_message = 'License bị từ chối [' . $code_text . ']: ' . $msg . $extra;
    return false;
}

        $max_posts  = isset( $data['max_posts'] ) ? intval( $data['max_posts'] ) : 0;
        $used_posts = isset( $data['used_posts'] ) ? intval( $data['used_posts'] ) : 0;
        $plan       = isset( $data['plan'] ) ? sanitize_text_field( $data['plan'] ) : '';

        if ( $max_posts <= 0 ) {
            $error_message = 'License hợp lệ nhưng max_posts <= 0.';
            return false;
        }

        $license_info = array(
            'max_posts'  => $max_posts,
            'used_posts' => $used_posts,
            'plan'       => $plan,
        );

        return true;
    }

    private function consume_remote_license_usage( $license, $count = 1 ) {
        $serial = $license['serial'] ?? '';
        $email  = $license['email'] ?? '';

        if ( empty( $serial ) || empty( $email ) ) {
            return $license;
        }

        $site_url = site_url();
$host = parse_url( $site_url, PHP_URL_HOST );
if ( ! $host ) {
    $host = $site_url;
}
// Chuẩn hoá domain: bỏ www, bỏ port, lowercase
$domain = strtolower( preg_replace( '/^www\./i', '', preg_replace( '/:\\d+$/', '', (string) $host ) ) );

$response = wp_remote_post(
            self::LICENSE_API_URL,
            array(
                'timeout' => 25,
                'body'    => array(
                    'action'   => 'consume',
                    'serial'   => $serial,
                    'email'    => $email,
                    'domain'   => $domain,
                    'site_url' => $site_url,
                    'count'    => max( 1, intval( $count ) ),
                ),
            )
        );

        if ( is_wp_error( $response ) ) {
            return $license;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        if ( $code < 200 || $code >= 300 ) {
            return $license;
        }

        $data = json_decode( $body, true );
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            return $license;
        }

        if ( isset( $data['status'] ) && $data['status'] === 'ok' ) {
            $license['used_posts'] = intval( $data['used_posts'] ?? ( $license['used_posts'] ?? 0 ) );
            $license['max_posts']  = intval( $data['max_posts']  ?? ( $license['max_posts']  ?? 0 ) );
            $license['plan']       = sanitize_text_field( $data['plan'] ?? ( $license['plan'] ?? '' ) );
        }

        return $license;
    }

    private function redirect_with_message( $msg ) {
        $url = add_query_arg(
            array(
                'page'          => 'abcgs-ai-bulk-silo',
                'abcgs_message' => rawurlencode( $msg ),
            ),
            admin_url( 'admin.php' )
        );
        wp_safe_redirect( $url );
        exit;
    }

    private function redirect_with_error( $msg ) {
        $url = add_query_arg(
            array(
                'page'        => 'abcgs-ai-bulk-silo',
                'abcgs_error' => rawurlencode( $msg ),
            ),
            admin_url( 'admin.php' )
        );
        wp_safe_redirect( $url );
        exit;
    }
}

new AI_Bulk_ChatGPT_Silo();
